import requests
import json
from http import HTTPMethod
from pathlib import Path

class APIUtils:
    def __init__(self, base_url: str = "http://127.0.0.1:8011"):
        self.base_url = base_url

    def _send_request(
            self, 
            method: HTTPMethod, 
            endpoint: str, 
            data: dict = None, 
            params: dict = None, 
            version: str = "1.0"
        ) -> dict:
        """
        Sends a request to the RTX Remix REST API and returns the response as a dictionary.

        Args:
            method: The HTTP method to use for the request (e.g. GET, POST, PUT, DELETE).
            endpoint: The endpoint to send the request to.
            data: The data to send in the request body.
            params: The query parameters to send with the request.
            version: The version of the API to use.
        """

        url = f"{self.base_url}{endpoint}"
        headers = {
            "Content-Type": "application/json",
            "Accept": f"application/lightspeed.remix.service+json; version={version}",
        }
        
        response = requests.request(
            method, url, headers=headers, data=json.dumps(data), params=params
        )
        response.raise_for_status()

        return response.json()
    
    def _encode_string(self, string: str) -> str:
        """
        Encode a string to be used in a URL.
        """
        return requests.utils.quote(string, safe='')

    def get_status(self) -> bool:
        """
        Get the status of the RTX Remix API.
        """
        try:
            self._send_request(HTTPMethod.GET, "/status")
            return True
        except Exception:
            return False


    def get_opened_project(self) -> str | None:
        """
        Get the currently opened project. Will return None if no project is opened.
        """
        try:
            response = self._send_request(HTTPMethod.GET, "/stagecraft/project")
            return response.get("layer_id")
        except requests.exceptions.HTTPError as e:
            # The REST API is expected to return a 404 if no project is opened
            if e.response.status_code == 404:
                return None
            else:
                raise

    def get_assets(self, **kwargs) -> list[str]:
        """
        Get the assets in the currently opened project.

        Args:
            **kwargs: Additional query parameters to send with the request.

        Possible Query Parameters:
        - asset_hashes: list[str] - A list of specific asset hashes to get 
        - asset_types: list[str] - A list of specific asset types to get
        - selection: bool - Whether to return only selected assets or all assets
        - filter_session_assets: bool - Whether to filter out assets defined on the session layer or not
        - layer_identifier: str - Look for assets that exists or not on a given layer
        - exists: bool - Whether to look for assets that exists or not on `layer_identifier`
        """
        params = {k: v for k, v in kwargs.items() if v is not None}
        response = self._send_request(HTTPMethod.GET, "/stagecraft/assets", params=params)

        return response.get("asset_paths")
    
    def get_file_paths(self, asset_path: str) -> list[str]:
        """
        Get the referenced absolute file paths for the given asset path.

        Args:
            asset_path: The asset path to get the referenced file paths for
        """
        encoded_path = self._encode_string(asset_path)
        response = self._send_request(HTTPMethod.GET, f"/stagecraft/assets/{encoded_path}/file-paths")
        reference_paths = response.get("reference_paths")

        # Build absolute paths from the reference paths
        file_paths = []
        for _, path in reference_paths:
            asset_path, layer_id = path
            file_paths.append(str(Path(layer_id).parent / asset_path))

        return file_paths
        
    def get_output_path(self) -> str:
        """
        Get the default ingested assets output path for the currently opened project.
        """
        response = self._send_request(HTTPMethod.GET, "/stagecraft/assets/default-directory")
        return response.get("asset_path")

    def ingest_models(
            self, input_paths: list[str], output_path: str, executor: int = 1, usd_extension: str = "usd"
        ) -> list[str]:
        """
        Ingest a Blender mesh.

        Args:
            input_paths: A list of absolute file paths to the meshes to ingest
            output_path: The output path to store the ingested assets in

        Returns:
            A list of absolute file paths to the ingested assets
        """

        if executor not in [0, 1]:
            raise ValueError(f"Unsupported executor: {executor}.")

        if usd_extension not in ["usd", "usda", "usdc"]:
            raise ValueError(f"Unsupported USD extension: {usd_extension}.")

        # Prepare the data for the API request
        data = {
            "executor": executor,
            "context_plugin": {
                "data": {
                    "input_files": input_paths,
                    "output_directory": output_path,
                    "output_usd_extension": usd_extension,
                },
            }
        }

        # Send the request
        response = self._send_request(HTTPMethod.POST, "/ingestcraft/mass-validator/queue/model", data=data)

        # Get the output paths when the ingestion is complete
        output_paths = []
        for completed_task in response.get("completed_schemas"):
            dataflows = completed_task.get("context_plugin").get("data").get("data_flows")
            for dataflow in dataflows:
                if dataflow.get("channel") != "ingestion_output":
                    continue
                output_paths.extend(dataflow.get("output_data"))

        return output_paths
        
    def append_file_path(self, prim_path: str, asset_file_path: str):
        """
        Replace an asset reference file path with a new one.
        """
        encoded_path = self._encode_string(prim_path)
        data = {
            "asset_file_path": asset_file_path
        }
        self._send_request(HTTPMethod.POST, f"/stagecraft/assets/{encoded_path}/file-paths", data=data)
