import bpy

from ..utils.api import APIUtils
from ..utils.prims import PrimsUtils


class REMIX_OT_ImportAssets(bpy.types.Operator):
    bl_idname = "rtx_remix.import_assets"
    bl_label = "Import Assets"
    bl_description = "Import the assets selected in the RTX Remix Toolkit"
    bl_options = {"REGISTER", "UNDO"}  # Enable undo for the operator.

    def __init__(self):
        self._api = APIUtils()

    def __del__(self):
        self._api = None

    def execute(self, context):
        # Get the properties
        props = context.scene.remix_import_props

        # Make sure the API is running
        if not self._api.get_status():
            self.report({"ERROR"}, "The REST API is not running")
            return {"CANCELLED"}
        
        # Make sure a project is opened
        if not self._api.get_opened_project():
            self.report({"ERROR"}, "No project is opened")
            return {"CANCELLED"}
        
        try:
            # Get the assets selected in the RTX Remix viewport
            assets = self._api.get_assets(selection=True, asset_types="models")

            # Get the mesh paths for the selected assets
            meshes = [PrimsUtils.get_mesh_path(asset) for asset in assets if PrimsUtils.get_mesh_path(asset)]

            # Get the referenced file paths for the selected assets
            file_paths = set()
            for mesh in meshes:
                file_paths = file_paths.union(self._api.get_file_paths(mesh))
            
            # Import the selected assets
            for file_path in file_paths:
                # Assets coming from Remix are always in USD format
                bpy.ops.wm.usd_import(
                    filepath=file_path,
                    create_collection=props.create_collection,
                    scale=props.assets_scale,
                    import_visible_only=False,
                    relative_path=False,
                )
        except Exception as e:
            self.report({"ERROR"}, str(e))
            return {"CANCELLED"}    

        return {"FINISHED"}
    