import bpy
from pathlib import Path
from tempfile import TemporaryDirectory

from ..utils.api import APIUtils
from ..utils.prims import PrimsUtils


class REMIX_OT_ExportAssets(bpy.types.Operator):
    bl_idname = "rtx_remix.export_assets"
    bl_label = "Export Assets"
    bl_description = (
        "Export the selected assets to the RTX Remix Toolkit and append the reference to the "
        "selected meshes"
    )

    def __init__(self):
        self._api = APIUtils()

    def __del__(self):
        self._api = None        

    def execute(self, context):
        props = context.scene.remix_export_props

        # Make sure an export target is selected
        if not props.export_target:
            self.report({"ERROR"}, "An export target must be selected")
            return {"CANCELLED"}
    
        # Make sure the API is running
        if not self._api.get_status():
            self.report({"ERROR"}, "The REST API is not running")
            return {"CANCELLED"}
        
        # Make sure a project is opened
        if not self._api.get_opened_project():
            self.report({"ERROR"}, "No project is opened")
            return {"CANCELLED"}
        
        try:
            # Get the assets selected in the RTX Remix viewport
            assets = self._api.get_assets(selection=True, asset_types="models")

            # Get the mesh paths for the selected assets
            meshes = [PrimsUtils.get_mesh_path(asset) for asset in assets if PrimsUtils.get_mesh_path(asset)]

            if not meshes:
                self.report({"ERROR"}, "No asset selected in RTX Remix")
                return {"CANCELLED"}
            
            # If multiple meshes are selected in Remix, use the first one
            target_mesh = meshes[0]

            # Get the ingestion output path
            output_path = self._api.get_output_path()

            # Selected the Export Target
            bpy.ops.object.select_all(action="DESELECT")
            props.export_target.select_set(True)
            context.view_layer.objects.active = props.export_target

            # Pack all textures
            bpy.ops.file.pack_all()
                
            # Setup a temporarary directory
            with TemporaryDirectory() as temp_dir:
                # Build a temporary file path
                temp_file = str((Path(temp_dir) / props.export_target.name).with_suffix(".obj"))
                
                # Export the target Object
                bpy.ops.wm.obj_export(
                    filepath=temp_file,
                    global_scale=props.assets_scale,
                    export_pbr_extensions=True,
                    path_mode="COPY",
                    export_triangulated_mesh=True,
                )

                # Ingest the exported model
                ingested_paths = self._api.ingest_models(
                    [temp_file], 
                    output_path, 
                    executor=int(props.ingestion_executor), 
                    usd_extension=props.usd_extension
                )

            # Append the references to the target mesh
            for ingested_path in ingested_paths:
                self._api.append_file_path(target_mesh, ingested_path)

        except Exception as e:
            self.report({"ERROR"}, str(e))
            return {"CANCELLED"}    

        return {"FINISHED"}
    